/*
Veidotājs: Roberts Blite
Mērķis: Šis skripts pārvalda materiālu tabulu - iegūst datus no servera, attēlo tos tabulā,
        pievieno jaunus materiālus, noņem materiālus un nodrošina meklēšanas funkcionalitāti.
Apraksts: Skripts izmanto fetch API, lai veiktu AJAX pieprasījumus uz serveri (`material-count.php`),
          atjauninātu materiālu tabulu un pārvaldītu materiālu pievienošanu/noņemšanu.
*/

// Gaida, līdz HTML dokuments ir pilnībā ielādēts
document.addEventListener("DOMContentLoaded", () => {
    const tableBody = document.querySelector(".table tbody"); // Materiālu tabulas saturs
    const addForm = document.getElementById("add-material-form"); // Pievienošanas forma
    const removeForm = document.getElementById("remove-material-form"); // Noņemšanas forma
    const searchBar = document.getElementById("search-bar"); // Meklēšanas josla

    // Funkcija, lai iegūtu materiālus no datubāzes
    const fetchMaterials = async () => {
        try {
            const response = await fetch("../php/material-count.php"); // API pieprasījums
            const materials = await response.json();

            if (Array.isArray(materials)) {
                renderTable(materials); // Attēlo materiālus tabulā
            } else if (materials.error) {
                alert(materials.error); // Parāda kļūdu, ja tāda ir
            }
        } catch (error) {
            console.error("Kļūda, iegūstot materiālus:", error);
        }
    };

    // Funkcija, lai attēlotu materiālu datus tabulā
    const renderTable = (materials) => {
        tableBody.innerHTML = ""; // Attīra tabulas saturu

        materials.forEach((material) => {
            // Nosaka statusa krāsu, pamatojoties uz materiāla stāvokli
            const statusClass =
                material.status === "Pietiekami"
                    ? "status-green"
                    : material.status === "Mazāk par normu"
                        ? "status-orange"
                        : "status-red";

            // Izveido tabulas rindu ar materiāla datiem
            const row = `
                <tr>
                    <td>${material.sku_code}</td>
                    <td>${material.material_type}</td>
                    <td>${material.material_size}</td>
                    <td>${material.material_count}</td>
                    <td class="${statusClass}">${material.status}</td>
                </tr>
            `;
            tableBody.insertAdjacentHTML("beforeend", row); // Pievieno jauno rindu tabulai
        });
    };

    // Funkcija, lai pievienotu jaunu materiālu
    addForm.addEventListener("submit", async (e) => {
        e.preventDefault(); // Novērš noklusējuma formu darbību

        const formData = new FormData(addForm); // Iegūst datus no formas
        const materialData = Object.fromEntries(formData.entries()); // Pārveido datus objektā

        try {
            const response = await fetch("../php/material-count.php?action=add", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify(materialData), // Nosūta datus JSON formātā
            });

            const result = await response.json();
            if (result.success) {
                alert(result.success); // Paziņo par veiksmīgu pievienošanu
                fetchMaterials(); // Atjauno materiālu sarakstu
                addForm.reset(); // Notīra formu
            } else if (result.error) {
                alert(result.error); // Parāda kļūdu
            }
        } catch (error) {
            console.error("Kļūda, pievienojot materiālu:", error);
        }
    });

    // Funkcija, lai noņemtu materiālu no datubāzes
    removeForm.addEventListener("submit", async (e) => {
        e.preventDefault(); // Novērš noklusējuma formu darbību

        if (!confirm("Vai tiešām vēlaties noņemt materiālu?")) return; // Apstiprinājums no lietotāja

        const formData = new FormData(removeForm); // Iegūst datus no formas
        const removeData = Object.fromEntries(formData.entries()); // Pārveido datus objektā

        try {
            const response = await fetch("../php/material-count.php?action=remove", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify(removeData), // Nosūta datus JSON formātā
            });

            const result = await response.json();
            if (result.success) {
                alert(result.success); // Paziņo par veiksmīgu noņemšanu
                fetchMaterials(); // Atjauno materiālu sarakstu
                removeForm.reset(); // Notīra formu
            } else if (result.error) {
                alert(result.error); // Parāda kļūdu
            }
        } catch (error) {
            console.error("Kļūda, noņemot materiālu:", error);
        }
    });

    // Meklēšanas funkcionalitāte
    searchBar.addEventListener("input", (e) => {
        const searchTerm = e.target.value.toLowerCase(); // Iegūst meklēšanas terminu
        const rows = tableBody.querySelectorAll("tr"); // Iegūst visas tabulas rindas

        // Filtrē rindas atbilstoši meklēšanas terminam
        rows.forEach((row) => {
            const cells = Array.from(row.querySelectorAll("td")); // Iegūst visas rindas šūnas
            const match = cells.some((cell) => cell.textContent.toLowerCase().includes(searchTerm));
            row.style.display = match ? "" : "none"; // Slēpj vai parāda rindu
        });
    });

    // Sākotnējā materiālu ielāde
    fetchMaterials();
});
